<?php

namespace App\Http\Controllers;

use App\Models\EmpresaProtocolo;
use App\Models\EmpresaProtocoloPeriodo;
use App\Models\Protocolo;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Carbon\Carbon;

class EmpresaProtocoloController extends Controller
{
    public function index()
    {
        $empresa = auth('empresa')->user();
        $protocolos = Protocolo::where('status', true)
            ->where('visivel_empresas', true)
            ->get();
        $empresaProtocolos = EmpresaProtocolo::with('periodos')
            ->where('empresa_id', $empresa->id)
            ->get()
            ->keyBy('protocolo_id');
        return view('empresas.protocolos.index', compact('protocolos', 'empresaProtocolos'));
    }

    public function toggle(Request $request, Protocolo $protocolo)
    {
        $empresa = auth('empresa')->user();

        if (!$protocolo->visivel_empresas) {
            abort(403);
        }

        $ativo = $request->boolean('ativo');

        if ($empresa->protocolos()->where('protocolo_id', $protocolo->id)->exists()) {
            $empresa->protocolos()->updateExistingPivot($protocolo->id, ['ativo' => $ativo]);
        } else {
            $empresa->protocolos()->attach($protocolo->id, ['ativo' => $ativo]);
        }

        return back();
    }

    public function storePeriodo(Request $request, Protocolo $protocolo)
    {
        $empresa = auth('empresa')->user();

        if (!$protocolo->visivel_empresas) {
            abort(403);
        }

        $validated = $request->validate([
            'inicio' => ['required', 'string'],
            'fim' => ['required', 'string'],
        ]);

        try {
            $inicio = Carbon::createFromFormat('d/m/Y', $validated['inicio']);
        } catch (\Exception $e) {
            return back()->withErrors(['inicio' => 'Formato de data inválido. Utilize dd/mm/aaaa.']);
        }

        try {
            $fim = Carbon::createFromFormat('d/m/Y', $validated['fim']);
        } catch (\Exception $e) {
            return back()->withErrors(['fim' => 'Formato de data inválido. Utilize dd/mm/aaaa.']);
        }

        if ($fim->lt($inicio)) {
            return back()->withErrors(['fim' => 'A data final deve ser posterior ou igual à data inicial.']);
        }

        $empresaProtocolo = EmpresaProtocolo::firstOrCreate(
            ['empresa_id' => $empresa->id, 'protocolo_id' => $protocolo->id],
            ['ativo' => true]
        );

        if (!$empresaProtocolo->ativo) {
            abort(403);
        }

        $empresaProtocolo->periodos()->create([
            'inicio' => $inicio->toDateString(),
            'fim' => $fim->toDateString(),
            'token' => (string) Str::uuid(),
        ]);

        return back();
    }

    public function destroyPeriodo(EmpresaProtocoloPeriodo $periodo)
    {
        $empresa = auth('empresa')->user();

        if ($periodo->empresaProtocolo->empresa_id !== $empresa->id) {
            abort(403);
        }

        $periodo->delete();

        return back();
    }

    public function storeCpfs(Request $request, EmpresaProtocoloPeriodo $periodo)
    {
        $empresa = auth('empresa')->user();

        if ($periodo->empresaProtocolo->empresa_id !== $empresa->id) {
            abort(403);
        }

        $cpfs = collect(preg_split("/\r\n|\n|\r/", $request->input('cpfs', '')))
            ->map(fn($cpf) => preg_replace('/\D/', '', $cpf))
            ->filter()
            ->unique()
            ->implode("\n");

        $periodo->update(['cpfs' => $cpfs]);

        return back();
    }
}
