<?php

namespace App\Http\Controllers;

use App\Models\Pergunta;
use App\Models\Protocolo;
use Illuminate\Http\Request;

class PerguntaController extends Controller
{
    public function store(Request $request, Protocolo $protocolo)
    {
        $validated = $request->validate([
            'texto' => 'required|string',
            'ordem' => 'nullable|integer',
            'opcoes' => 'required|array|min:1',
            'opcoes.*.texto' => 'required|string',
            'opcoes.*.pontuacao' => 'required|integer',
        ]);

        $pergunta = $protocolo->perguntas()->create([
            'texto' => $validated['texto'],
            'ordem' => $validated['ordem'] ?? null,
        ]);

        foreach ($validated['opcoes'] as $opcao) {
            $pergunta->opcoes()->create($opcao);
        }

        return redirect()->route('protocolos.perguntas', $protocolo)->with('success', 'Pergunta adicionada com sucesso.');
    }

    public function edit(Protocolo $protocolo, Pergunta $pergunta)
    {
        if ($pergunta->protocolo_id !== $protocolo->id) {
            abort(404);
        }

        $pergunta->load('opcoes');

        return view('protocolos.perguntas-edit', compact('protocolo', 'pergunta'));
    }

    public function update(Request $request, Protocolo $protocolo, Pergunta $pergunta)
    {
        if ($pergunta->protocolo_id !== $protocolo->id) {
            abort(404);
        }

        $validated = $request->validate([
            'texto' => 'required|string',
            'ordem' => 'nullable|integer',
            'opcoes' => 'required|array|min:1',
            'opcoes.*.id' => 'nullable|integer',
            'opcoes.*.texto' => 'required|string',
            'opcoes.*.pontuacao' => 'required|integer',
        ]);

        $pergunta->update([
            'texto' => $validated['texto'],
            'ordem' => $validated['ordem'] ?? $pergunta->ordem,
        ]);

        $existingIds = $pergunta->opcoes()->pluck('id')->toArray();
        $submittedIds = [];

        foreach ($validated['opcoes'] as $opcao) {
            if (! empty($opcao['id'])) {
                $submittedIds[] = $opcao['id'];
                $pergunta->opcoes()->where('id', $opcao['id'])->update([
                    'texto' => $opcao['texto'],
                    'pontuacao' => $opcao['pontuacao'],
                ]);
            } else {
                $pergunta->opcoes()->create([
                    'texto' => $opcao['texto'],
                    'pontuacao' => $opcao['pontuacao'],
                ]);
            }
        }

        $idsToDelete = array_diff($existingIds, $submittedIds);
        if (! empty($idsToDelete)) {
            $pergunta->opcoes()->whereIn('id', $idsToDelete)->delete();
        }

        return redirect()->route('protocolos.perguntas', $protocolo)->with('success', 'Pergunta atualizada com sucesso.');
    }

    public function destroy(Protocolo $protocolo, Pergunta $pergunta)
    {
        if ($pergunta->protocolo_id !== $protocolo->id) {
            abort(404);
        }

        $pergunta->delete();

        return redirect()->route('protocolos.perguntas', $protocolo)->with('success', 'Pergunta removida com sucesso.');
    }
}
